import { ReactNode } from "react";
import { Link, useLocation } from "wouter";
import { useAuth } from "@/context/AuthContext";
import { 
  LayoutDashboard, 
  Megaphone, 
  Wallet, 
  MessageSquare, 
  Settings, 
  Users, 
  FileText,
  BarChart3,
  LogOut,
  Menu,
  BrainCircuit
} from "lucide-react";
import { Button } from "@/components/ui/button";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { Separator } from "@/components/ui/separator";

interface SidebarLink {
  name: string;
  href: string;
  icon: any;
}

const BRAND_LINKS: SidebarLink[] = [
  { name: "Overview", href: "/dashboard", icon: LayoutDashboard },
  { name: "Campaigns", href: "/dashboard/campaigns", icon: Megaphone },
  { name: "AI Matcher", href: "/algorithm", icon: BrainCircuit },
  { name: "Messages", href: "/dashboard/messages", icon: MessageSquare },
  { name: "Wallet", href: "/dashboard/wallet", icon: Wallet },
  { name: "Settings", href: "/dashboard/settings", icon: Settings },
];

const INFLUENCER_LINKS: SidebarLink[] = [
  { name: "Overview", href: "/dashboard", icon: LayoutDashboard },
  { name: "My Jobs", href: "/dashboard/jobs", icon: BriefcaseIcon },
  { name: "Portfolio", href: "/profile", icon: Users },
  { name: "Earnings", href: "/dashboard/wallet", icon: Wallet },
  { name: "Messages", href: "/dashboard/messages", icon: MessageSquare },
  { name: "Settings", href: "/dashboard/settings", icon: Settings },
];

const ADMIN_LINKS: SidebarLink[] = [
  { name: "Overview", href: "/dashboard", icon: LayoutDashboard },
  { name: "Users", href: "/dashboard/users", icon: Users },
  { name: "Disputes", href: "/dashboard/disputes", icon: FileText },
  { name: "Financials", href: "/dashboard/financials", icon: BarChart3 },
  { name: "Settings", href: "/dashboard/settings", icon: Settings },
];

// Helper for icon
function BriefcaseIcon(props: any) {
  return <Megaphone {...props} />; 
}

export function DashboardLayout({ children }: { children: ReactNode }) {
  const { user, logout } = useAuth();
  const [location] = useLocation();

  if (!user) return <div className="p-10 text-center">Please log in</div>;

  const links = user.role === "influencer" 
    ? INFLUENCER_LINKS 
    : user.role === "admin" 
      ? ADMIN_LINKS 
      : BRAND_LINKS;

  const SidebarContent = () => (
    <div className="flex flex-col h-full bg-sidebar text-sidebar-foreground border-r border-sidebar-border">
      <div className="h-16 flex items-center px-6 border-b border-sidebar-border">
        <Link href="/">
          <div className="flex items-center gap-2 cursor-pointer">
             <div className="size-7 rounded-md bg-primary flex items-center justify-center text-white font-bold text-lg">I</div>
             <span className="font-display font-bold text-lg">InfluencerHub</span>
          </div>
        </Link>
      </div>

      <div className="flex-1 py-6 px-3 space-y-1">
        {links.map((link) => {
          const isActive = location === link.href;
          return (
            <Link key={link.name} href={link.href}>
              <div className={`flex items-center gap-3 px-3 py-2.5 rounded-md text-sm font-medium transition-colors cursor-pointer ${isActive ? 'bg-sidebar-accent text-sidebar-accent-foreground' : 'text-muted-foreground hover:bg-sidebar-accent/50 hover:text-sidebar-foreground'}`}>
                <link.icon className="size-4" />
                {link.name}
              </div>
            </Link>
          );
        })}
      </div>

      <div className="p-4 border-t border-sidebar-border">
        <div className="flex items-center gap-3 mb-4 px-2">
          <Avatar className="size-8">
            <AvatarImage src={user.avatar} />
            <AvatarFallback>{user.name.charAt(0)}</AvatarFallback>
          </Avatar>
          <div className="overflow-hidden">
            <p className="text-sm font-medium truncate">{user.name}</p>
            <p className="text-xs text-muted-foreground capitalize">{user.role}</p>
          </div>
        </div>
        <Button variant="outline" className="w-full justify-start text-muted-foreground" onClick={logout}>
          <LogOut className="mr-2 size-4" />
          Log out
        </Button>
      </div>
    </div>
  );

  return (
    <div className="flex h-screen bg-background">
      {/* Desktop Sidebar */}
      <div className="hidden lg:block w-64 shrink-0">
        <SidebarContent />
      </div>

      {/* Mobile Sidebar */}
      <div className="lg:hidden absolute top-4 left-4 z-50">
        <Sheet>
          <SheetTrigger asChild>
            <Button variant="outline" size="icon">
              <Menu className="size-5" />
            </Button>
          </SheetTrigger>
          <SheetContent side="left" className="p-0 w-64">
            <SidebarContent />
          </SheetContent>
        </Sheet>
      </div>

      {/* Main Content */}
      <main className="flex-1 overflow-y-auto">
        <div className="container mx-auto p-4 lg:p-8 max-w-7xl">
          {children}
        </div>
      </main>
    </div>
  );
}
