import { Link, useLocation } from "wouter";
import { Button } from "@/components/ui/button";
import { useAuth } from "@/context/AuthContext";
import { useLanguage } from "@/context/LanguageContext";
import { 
  LayoutDashboard, 
  LogOut, 
  Menu, 
  Wallet, 
  Globe
} from "lucide-react";
import { useState } from "react";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import {
  DropdownMenu,
  DropdownMenuContent,
  DropdownMenuItem,
  DropdownMenuLabel,
  DropdownMenuSeparator,
  DropdownMenuTrigger,
} from "@/components/ui/dropdown-menu";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";

export function Navbar() {
  const [location] = useLocation();
  const { user, logout } = useAuth();
  const { t, language, setLanguage } = useLanguage();
  const [isOpen, setIsOpen] = useState(false);

  const isLanding = location === "/";

  const navLinks = [
    { name: t('nav.marketplace'), href: "/marketplace" },
    { name: t('nav.howItWorks'), href: "/#how-it-works" },
    { name: t('nav.algorithm'), href: "/algorithm" },
    { name: t('nav.pricing'), href: "/pricing" },
  ];

  const toggleLanguage = () => {
    setLanguage(language === 'en' ? 'es' : 'en');
  };

  return (
    <nav className={`w-full z-50 border-b transition-all duration-200 ${isLanding ? 'fixed top-0 left-0 right-0 bg-white/80 backdrop-blur-md border-transparent' : 'sticky top-0 bg-background/95 backdrop-blur-md border-border'}`}>
      <div className="container mx-auto px-4 h-16 flex items-center justify-between">
        {/* Logo */}
        <Link href="/">
          <div className="flex items-center gap-2 cursor-pointer">
            <div className="size-8 rounded-lg bg-primary flex items-center justify-center text-white font-bold text-xl">
              I
            </div>
            <span className="font-display font-bold text-xl tracking-tight">
              Influencer<span className="text-primary">Hub</span> AI
            </span>
          </div>
        </Link>

        {/* Desktop Nav */}
        <div className="hidden md:flex items-center gap-8">
          {navLinks.map((link) => (
            <Link key={link.name} href={link.href}>
              <span className="text-sm font-medium text-muted-foreground hover:text-primary transition-colors cursor-pointer">
                {link.name}
              </span>
            </Link>
          ))}
        </div>

        {/* Auth / Actions */}
        <div className="hidden md:flex items-center gap-4">
          {/* Language Toggle */}
          <Button variant="ghost" size="icon" onClick={toggleLanguage} className="rounded-full">
             <Globe className="size-4" />
             <span className="sr-only">Toggle Language</span>
             <span className="absolute -top-1 -right-1 text-[10px] font-bold bg-primary text-white px-1 rounded-full uppercase">{language}</span>
          </Button>

          {user ? (
            <div className="flex items-center gap-4">
              {user.balance !== undefined && (
                 <div className="flex items-center gap-2 px-3 py-1.5 bg-secondary rounded-full text-sm font-medium text-secondary-foreground">
                    <Wallet className="size-4 text-primary" />
                    <span>${user.balance.toLocaleString()}</span>
                 </div>
              )}
             
              <DropdownMenu>
                <DropdownMenuTrigger asChild>
                  <Button variant="ghost" className="relative h-8 w-8 rounded-full">
                    <Avatar className="h-8 w-8">
                      <AvatarImage src={user.avatar} alt={user.name} />
                      <AvatarFallback>{user.name.charAt(0)}</AvatarFallback>
                    </Avatar>
                  </Button>
                </DropdownMenuTrigger>
                <DropdownMenuContent className="w-56" align="end" forceMount>
                  <DropdownMenuLabel className="font-normal">
                    <div className="flex flex-col space-y-1">
                      <p className="text-sm font-medium leading-none">{user.name}</p>
                      <p className="text-xs leading-none text-muted-foreground">
                        {user.email}
                      </p>
                    </div>
                  </DropdownMenuLabel>
                  <DropdownMenuSeparator />
                  <DropdownMenuItem asChild>
                    <Link href="/dashboard">
                      <LayoutDashboard className="mr-2 h-4 w-4" />
                      <span>{t('nav.dashboard')}</span>
                    </Link>
                  </DropdownMenuItem>
                  <DropdownMenuItem asChild>
                    <Link href="/profile">
                      <div className="flex items-center">
                        <span className="mr-2">👤</span>
                        <span>{t('nav.profile')}</span>
                      </div>
                    </Link>
                  </DropdownMenuItem>
                  <DropdownMenuSeparator />
                  <DropdownMenuItem onClick={logout}>
                    <LogOut className="mr-2 h-4 w-4" />
                    <span>{t('nav.logout')}</span>
                  </DropdownMenuItem>
                </DropdownMenuContent>
              </DropdownMenu>
            </div>
          ) : (
            <>
              <Link href="/auth">
                <Button variant="ghost" size="sm">
                  {t('nav.login')}
                </Button>
              </Link>
              <Link href="/auth?tab=register">
                <Button size="sm" className="bg-primary hover:bg-primary/90 text-white shadow-lg shadow-primary/20">
                  {t('nav.getStarted')}
                </Button>
              </Link>
            </>
          )}
        </div>

        {/* Mobile Menu */}
        <Sheet open={isOpen} onOpenChange={setIsOpen}>
          <SheetTrigger asChild className="md:hidden">
            <Button variant="ghost" size="icon">
              <Menu className="size-6" />
            </Button>
          </SheetTrigger>
          <SheetContent side="right">
            <div className="flex flex-col gap-6 mt-8">
              <Button variant="outline" onClick={toggleLanguage} className="justify-start gap-2">
                 <Globe className="size-4" /> Switch to {language === 'en' ? 'Español' : 'English'}
              </Button>
              {navLinks.map((link) => (
                <Link key={link.name} href={link.href} onClick={() => setIsOpen(false)}>
                  <span className="text-lg font-medium block py-2 border-b border-border/50">
                    {link.name}
                  </span>
                </Link>
              ))}
              {!user && (
                <div className="flex flex-col gap-4 mt-4">
                  <Link href="/auth" onClick={() => setIsOpen(false)}>
                    <Button variant="outline" className="w-full">{t('nav.login')}</Button>
                  </Link>
                  <Link href="/auth?tab=register" onClick={() => setIsOpen(false)}>
                    <Button className="w-full">{t('nav.getStarted')}</Button>
                  </Link>
                </div>
              )}
            </div>
          </SheetContent>
        </Sheet>
      </div>
    </nav>
  );
}
