import { Influencer } from "@/lib/mock-data";
import { Card, CardContent, CardFooter, CardHeader } from "@/components/ui/card";
import { Badge } from "@/components/ui/badge";
import { Button } from "@/components/ui/button";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { CheckCircle2, Instagram, Youtube, Twitch, Twitter, Star } from "lucide-react";
import { Link } from "wouter";

const PlatformIcon = ({ platform }: { platform: Influencer["platform"] }) => {
  switch (platform) {
    case "Instagram": return <Instagram className="size-4 text-pink-600" />;
    case "YouTube": return <Youtube className="size-4 text-red-600" />;
    case "Twitch": return <Twitch className="size-4 text-purple-600" />;
    case "Twitter": return <Twitter className="size-4 text-blue-400" />;
    default: return null;
  }
};

export function InfluencerCard({ influencer }: { influencer: Influencer }) {
  return (
    <Card className="group overflow-hidden hover:shadow-xl transition-all duration-300 border-border/50 bg-card/50 backdrop-blur-sm">
      <div className="relative h-48 overflow-hidden">
        <div className="absolute inset-0 bg-gradient-to-t from-black/60 to-transparent z-10" />
        <img 
          src={influencer.image} 
          alt={influencer.name} 
          className="w-full h-full object-cover group-hover:scale-105 transition-transform duration-500"
        />
        <div className="absolute top-3 right-3 z-20">
          <Badge variant="secondary" className="backdrop-blur-md bg-white/90 text-black font-semibold shadow-sm">
            ${influencer.price}+
          </Badge>
        </div>
        <div className="absolute bottom-3 left-3 z-20 text-white">
          <div className="flex items-center gap-1.5 mb-1">
             <h3 className="font-display font-bold text-lg leading-none">{influencer.name}</h3>
             {influencer.verified && <CheckCircle2 className="size-4 text-blue-400 fill-blue-400/20" />}
          </div>
          <p className="text-xs text-white/80 font-medium">@{influencer.handle.replace('@', '')}</p>
        </div>
      </div>

      <CardContent className="pt-4 pb-2">
        <div className="flex justify-between items-start mb-4">
           <div className="flex flex-col">
             <span className="text-xs text-muted-foreground uppercase tracking-wider font-semibold">Followers</span>
             <span className="font-bold font-display text-lg">
               {(influencer.followers / 1000000).toFixed(1)}M
             </span>
           </div>
           <div className="flex flex-col text-right">
             <span className="text-xs text-muted-foreground uppercase tracking-wider font-semibold">Engagement</span>
             <span className="font-bold font-display text-lg text-green-600">
               {influencer.engagement}%
             </span>
           </div>
        </div>

        <div className="flex flex-wrap gap-2 mb-2">
           <Badge variant="outline" className="flex items-center gap-1 bg-secondary/30">
             <PlatformIcon platform={influencer.platform} />
             {influencer.platform}
           </Badge>
           <Badge variant="outline" className="bg-secondary/30 text-muted-foreground">
             {influencer.niche}
           </Badge>
        </div>
      </CardContent>

      <CardFooter className="pt-2 pb-4 flex gap-2">
        <Link href={`/influencer/${influencer.id}`} className="w-full">
          <Button variant="default" className="w-full bg-primary/10 text-primary hover:bg-primary hover:text-white border-0 shadow-none hover:shadow-md">
            View Profile
          </Button>
        </Link>
      </CardFooter>
    </Card>
  );
}
