import { createContext, useContext, useState, ReactNode } from "react";

export type UserRole = "guest" | "influencer" | "brand" | "admin" | "agency";

interface User {
  id: string;
  name: string;
  email: string;
  role: UserRole;
  avatar: string;
  balance?: number; // USD
}

interface AuthContextType {
  user: User | null;
  login: (role: UserRole) => void;
  logout: () => void;
  isLoading: boolean;
}

const AuthContext = createContext<AuthContextType | undefined>(undefined);

// Mock Users
const MOCK_USERS: Record<string, User> = {
  influencer: {
    id: "inf-1",
    name: "Sarah Jenkins",
    email: "sarah@example.com",
    role: "influencer",
    avatar: "https://images.unsplash.com/photo-1494790108377-be9c29b29330?w=150",
    balance: 1250.00,
  },
  brand: {
    id: "brd-1",
    name: "TechNova Inc.",
    email: "marketing@technova.com",
    role: "brand",
    avatar: "https://images.unsplash.com/photo-1560250097-0b93528c311a?w=150",
    balance: 5000.00,
  },
  admin: {
    id: "adm-1",
    name: "Super Admin",
    email: "admin@influencerhub.ai",
    role: "admin",
    avatar: "https://github.com/shadcn.png",
  },
  agency: {
    id: "agc-1",
    name: "Viral Loops Agency",
    email: "contact@viralloops.com",
    role: "agency",
    avatar: "https://images.unsplash.com/photo-1572044162444-ad6021105507?w=150",
    balance: 15000.00
  }
};

export function AuthProvider({ children }: { children: ReactNode }) {
  const [user, setUser] = useState<User | null>(null);
  const [isLoading, setIsLoading] = useState(false);

  const login = (role: UserRole) => {
    setIsLoading(true);
    // Simulate API delay
    setTimeout(() => {
      if (role === "guest") {
        setUser(null);
      } else {
        setUser(MOCK_USERS[role] || MOCK_USERS.brand);
      }
      setIsLoading(false);
    }, 600);
  };

  const logout = () => {
    setUser(null);
  };

  return (
    <AuthContext.Provider value={{ user, login, logout, isLoading }}>
      {children}
    </AuthContext.Provider>
  );
}

export function useAuth() {
  const context = useContext(AuthContext);
  if (context === undefined) {
    throw new Error("useAuth must be used within an AuthProvider");
  }
  return context;
}
