import { useState } from "react";
import { useLocation } from "wouter";
import { Navbar } from "@/components/layout/Navbar";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Textarea } from "@/components/ui/textarea";
import { Card, CardContent, CardFooter, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from "@/components/ui/select";
import { Slider } from "@/components/ui/slider";
import { CheckCircle2, ChevronRight, ChevronLeft, Rocket } from "lucide-react";
import { toast } from "@/hooks/use-toast";

export default function CreateCampaignPage() {
  const [, setLocation] = useLocation();
  const [step, setStep] = useState(1);
  const [loading, setLoading] = useState(false);

  const nextStep = () => setStep(s => s + 1);
  const prevStep = () => setStep(s => s - 1);

  const handleSubmit = () => {
    setLoading(true);
    setTimeout(() => {
      toast({
        title: "Campaign Created Successfully!",
        description: "Our AI is now matching you with the best influencers.",
      });
      setLoading(false);
      setLocation("/dashboard");
    }, 1500);
  };

  return (
    <div className="min-h-screen bg-background flex flex-col">
      <Navbar />
      
      <div className="flex-1 container mx-auto px-4 py-12 max-w-3xl">
        <div className="mb-8">
           <div className="flex items-center justify-between mb-4">
             <h1 className="text-3xl font-display font-bold">Create New Campaign</h1>
             <span className="text-sm text-muted-foreground">Step {step} of 3</span>
           </div>
           <div className="w-full h-2 bg-secondary rounded-full overflow-hidden">
             <div 
               className="h-full bg-primary transition-all duration-500 ease-out" 
               style={{ width: `${(step / 3) * 100}%` }} 
             />
           </div>
        </div>

        <Card className="shadow-xl border-border/50">
          {step === 1 && (
            <div className="animate-in slide-in-from-right-4 fade-in duration-300">
              <CardHeader>
                <CardTitle>Campaign Basics</CardTitle>
                <CardDescription>Define your goals and what you're promoting.</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="space-y-2">
                  <Label>Campaign Name</Label>
                  <Input placeholder="e.g. Summer 2025 Collection Launch" />
                </div>
                <div className="space-y-2">
                  <Label>Objective</Label>
                  <Select>
                    <SelectTrigger>
                      <SelectValue placeholder="Select goal" />
                    </SelectTrigger>
                    <SelectContent>
                      <SelectItem value="awareness">Brand Awareness</SelectItem>
                      <SelectItem value="sales">Sales / Conversions</SelectItem>
                      <SelectItem value="content">Content Creation</SelectItem>
                    </SelectContent>
                  </Select>
                </div>
                <div className="space-y-2">
                  <Label>Campaign Brief</Label>
                  <Textarea 
                    placeholder="Describe your product, key messages, and deliverables..." 
                    className="min-h-[120px]"
                  />
                </div>
              </CardContent>
              <CardFooter className="flex justify-end">
                <Button onClick={nextStep}>
                  Next Step <ChevronRight className="ml-2 size-4" />
                </Button>
              </CardFooter>
            </div>
          )}

          {step === 2 && (
            <div className="animate-in slide-in-from-right-4 fade-in duration-300">
              <CardHeader>
                <CardTitle>Targeting & Budget</CardTitle>
                <CardDescription>Who do you want to reach?</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="grid grid-cols-2 gap-4">
                   <div className="space-y-2">
                     <Label>Target Niche</Label>
                     <Select>
                        <SelectTrigger>
                          <SelectValue placeholder="Select niche" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="fashion">Fashion</SelectItem>
                          <SelectItem value="tech">Tech</SelectItem>
                          <SelectItem value="gaming">Gaming</SelectItem>
                        </SelectContent>
                     </Select>
                   </div>
                   <div className="space-y-2">
                     <Label>Platform</Label>
                     <Select>
                        <SelectTrigger>
                          <SelectValue placeholder="Select platform" />
                        </SelectTrigger>
                        <SelectContent>
                          <SelectItem value="instagram">Instagram</SelectItem>
                          <SelectItem value="tiktok">TikTok</SelectItem>
                          <SelectItem value="youtube">YouTube</SelectItem>
                        </SelectContent>
                     </Select>
                   </div>
                </div>

                <div className="space-y-4">
                  <div className="flex justify-between">
                    <Label>Total Budget (USD)</Label>
                    <span className="font-bold font-mono text-primary">$5,000</span>
                  </div>
                  <Slider defaultValue={[5000]} max={50000} step={100} />
                  <p className="text-xs text-muted-foreground">Funds will be held in escrow upon matching.</p>
                </div>
              </CardContent>
              <CardFooter className="flex justify-between">
                <Button variant="ghost" onClick={prevStep}>
                  <ChevronLeft className="mr-2 size-4" /> Back
                </Button>
                <Button onClick={nextStep}>
                  Next Step <ChevronRight className="ml-2 size-4" />
                </Button>
              </CardFooter>
            </div>
          )}

          {step === 3 && (
            <div className="animate-in slide-in-from-right-4 fade-in duration-300">
              <CardHeader>
                <CardTitle>Review & Launch</CardTitle>
                <CardDescription>Ready to blast off?</CardDescription>
              </CardHeader>
              <CardContent className="space-y-6">
                <div className="bg-secondary/30 p-6 rounded-lg border border-border/50 space-y-4">
                  <div className="flex items-center gap-4">
                    <div className="size-12 rounded-full bg-green-100 flex items-center justify-center">
                      <Rocket className="size-6 text-green-600" />
                    </div>
                    <div>
                      <h4 className="font-bold text-lg">Ready to launch</h4>
                      <p className="text-sm text-muted-foreground">Our AI will instantly analyze 15,000+ influencers to find your match.</p>
                    </div>
                  </div>
                  
                  <div className="space-y-2 pt-4 border-t border-border/50">
                    <div className="flex justify-between text-sm">
                      <span className="text-muted-foreground">Campaign Name:</span>
                      <span className="font-medium">Summer 2025 Launch</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span className="text-muted-foreground">Budget:</span>
                      <span className="font-medium">$5,000.00</span>
                    </div>
                    <div className="flex justify-between text-sm">
                      <span className="text-muted-foreground">Objective:</span>
                      <span className="font-medium">Sales / Conversions</span>
                    </div>
                  </div>
                </div>

                <div className="flex items-start gap-2 text-sm text-muted-foreground">
                  <CheckCircle2 className="size-4 text-primary mt-0.5" />
                  <p>I agree to the terms of service and authorize the escrow deposit.</p>
                </div>
              </CardContent>
              <CardFooter className="flex justify-between">
                <Button variant="ghost" onClick={prevStep}>
                  <ChevronLeft className="mr-2 size-4" /> Back
                </Button>
                <Button onClick={handleSubmit} disabled={loading} className="bg-green-600 hover:bg-green-700 text-white shadow-lg shadow-green-600/20">
                  {loading ? "Launching..." : "Launch Campaign"}
                </Button>
              </CardFooter>
            </div>
          )}
        </Card>
      </div>
    </div>
  );
}
