import { useState } from "react";
import { useRoute } from "wouter";
import { influencers } from "@/lib/mock-data";
import { Navbar } from "@/components/layout/Navbar";
import { Footer } from "@/components/layout/Footer";
import { Button } from "@/components/ui/button";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Card, CardContent, CardHeader, CardTitle } from "@/components/ui/card";
import { Separator } from "@/components/ui/separator";
import { CheckCircle2, MapPin, MessageSquare, Share2, Star } from "lucide-react";
import { useLanguage } from "@/context/LanguageContext";

export default function InfluencerProfilePage() {
  const { t } = useLanguage();
  const [match, params] = useRoute("/influencer/:id");
  const influencer = influencers.find(inf => inf.id === params?.id);

  if (!influencer) return <div>Influencer not found</div>;

  return (
    <div className="min-h-screen bg-background">
      <Navbar />

      {/* Header Banner */}
      <div className="h-64 md:h-80 bg-gradient-to-r from-indigo-500 to-purple-600 relative">
        <div className="absolute inset-0 bg-black/10" />
      </div>

      <div className="container mx-auto px-4 -mt-20 relative z-10 pb-20">
        <div className="grid lg:grid-cols-3 gap-8">
          
          {/* Left Column: Profile Info */}
          <div className="lg:col-span-2 space-y-8">
            <div className="bg-card rounded-2xl p-6 shadow-lg border border-border/50 relative">
               <div className="flex flex-col md:flex-row gap-6 items-start">
                 <div className="size-32 md:size-40 rounded-2xl overflow-hidden border-4 border-background shadow-md shrink-0">
                   <img src={influencer.image} alt={influencer.name} className="w-full h-full object-cover" />
                 </div>
                 
                 <div className="flex-1 space-y-2 pt-2">
                   <div className="flex flex-wrap items-center justify-between gap-4">
                     <div>
                       <h1 className="text-3xl font-display font-bold flex items-center gap-2">
                         {influencer.name} 
                         {influencer.verified && <CheckCircle2 className="size-6 text-blue-500" />}
                       </h1>
                       <p className="text-lg text-muted-foreground font-medium">{influencer.handle}</p>
                     </div>
                     <div className="flex gap-2">
                       <Button variant="outline" size="icon"><Share2 className="size-4" /></Button>
                       <Button variant="outline" size="icon"><MessageSquare className="size-4" /></Button>
                     </div>
                   </div>

                   <div className="flex flex-wrap gap-4 text-sm text-muted-foreground mt-2">
                     <div className="flex items-center gap-1">
                       <MapPin className="size-4" /> {influencer.location}
                     </div>
                     <div className="flex items-center gap-1">
                       <Star className="size-4 text-yellow-500 fill-yellow-500" /> {influencer.rating} ({influencer.reviews} {t('profile.reviews')})
                     </div>
                   </div>

                   <p className="text-foreground/80 leading-relaxed max-w-2xl pt-4">
                     {influencer.bio}
                   </p>

                   <div className="flex flex-wrap gap-2 pt-4">
                     <Badge variant="secondary">{influencer.niche}</Badge>
                     <Badge variant="outline">{influencer.language}</Badge>
                     <Badge variant="outline">{influencer.platform}</Badge>
                   </div>
                 </div>
               </div>
            </div>

            {/* Stats */}
            <div className="grid grid-cols-2 md:grid-cols-4 gap-4">
              <StatBox label={t('profile.followers')} value={`${(influencer.followers / 1000000).toFixed(1)}M`} />
              <StatBox label={t('marketplace.engagement')} value={`${influencer.engagement}%`} highlight />
              <StatBox label={t('profile.avgViews')} value="450K" />
              <StatBox label={t('profile.completedJobs')} value="84" />
            </div>

            {/* Content Tabs */}
            <Tabs defaultValue="portfolio" className="w-full">
              <TabsList className="w-full justify-start h-12 bg-transparent border-b rounded-none p-0 gap-8">
                <TabsTrigger value="portfolio" className="rounded-none border-b-2 border-transparent data-[state=active]:border-primary data-[state=active]:bg-transparent px-0 pb-2">{t('profile.portfolio')}</TabsTrigger>
                <TabsTrigger value="services" className="rounded-none border-b-2 border-transparent data-[state=active]:border-primary data-[state=active]:bg-transparent px-0 pb-2">{t('profile.services')}</TabsTrigger>
                <TabsTrigger value="reviews" className="rounded-none border-b-2 border-transparent data-[state=active]:border-primary data-[state=active]:bg-transparent px-0 pb-2">{t('profile.reviews')}</TabsTrigger>
              </TabsList>
              
              <TabsContent value="portfolio" className="pt-6">
                <div className="grid grid-cols-2 md:grid-cols-3 gap-4">
                  {[1, 2, 3, 4, 5, 6].map((i) => (
                    <div key={i} className="aspect-square rounded-lg bg-secondary/50 overflow-hidden hover:opacity-90 cursor-pointer">
                      <img 
                        src={`https://images.unsplash.com/photo-${1500000000000 + i}?w=400&h=400&fit=crop`} 
                        alt="Portfolio" 
                        className="w-full h-full object-cover"
                      />
                    </div>
                  ))}
                </div>
              </TabsContent>

              <TabsContent value="services" className="pt-6 space-y-4">
                 <ServiceCard title="Instagram Reel" price={influencer.price} desc="60s vertical video, posted to feed, link in bio for 24h." btnText={t('profile.select')} />
                 <ServiceCard title="Instagram Story Set" price={Math.round(influencer.price * 0.4)} desc="3 frames of stories, sticker link, mention." btnText={t('profile.select')} />
                 <ServiceCard title="Dedicated Post" price={Math.round(influencer.price * 0.8)} desc="High quality photo post, detailed caption, tags." btnText={t('profile.select')} />
              </TabsContent>
            </Tabs>
          </div>

          {/* Right Column: Action Card */}
          <div className="lg:col-span-1 space-y-6">
            <div className="sticky top-24">
              <Card className="shadow-xl border-primary/20 overflow-hidden">
                <div className="bg-primary/10 p-4 border-b border-primary/10">
                  <h3 className="font-bold text-primary">{t('profile.verified')}</h3>
                </div>
                <CardHeader>
                  <CardTitle className="text-3xl font-bold">${influencer.price}</CardTitle>
                  <p className="text-muted-foreground">{t('profile.startingPrice')}</p>
                </CardHeader>
                <CardContent className="space-y-4">
                  <ul className="space-y-2 text-sm">
                    <li className="flex items-center gap-2">
                      <CheckCircle2 className="size-4 text-green-500" /> {t('profile.verifiedAudience')}
                    </li>
                    <li className="flex items-center gap-2">
                      <CheckCircle2 className="size-4 text-green-500" /> {t('profile.highEngagement')}
                    </li>
                    <li className="flex items-center gap-2">
                      <CheckCircle2 className="size-4 text-green-500" /> {t('profile.fastResponse')}
                    </li>
                  </ul>
                  <Separator />
                  <div className="space-y-2">
                     <Button className="w-full h-12 text-lg font-bold shadow-lg shadow-primary/20">
                       {t('profile.bookNow')}
                     </Button>
                     <Button variant="outline" className="w-full">
                       {t('profile.contact')}
                     </Button>
                  </div>
                </CardContent>
                <div className="p-4 bg-secondary/30 text-xs text-center text-muted-foreground">
                  {t('profile.escrowNote')}
                </div>
              </Card>
            </div>
          </div>

        </div>
      </div>
      
      <Footer />
    </div>
  );
}

function StatBox({ label, value, highlight }: any) {
  return (
    <div className="bg-card p-4 rounded-xl border border-border/50 text-center shadow-sm">
      <p className="text-xs text-muted-foreground uppercase tracking-wider font-semibold mb-1">{label}</p>
      <p className={`text-2xl font-display font-bold ${highlight ? 'text-green-600' : 'text-foreground'}`}>{value}</p>
    </div>
  );
}

function ServiceCard({ title, price, desc, btnText }: any) {
  return (
    <div className="flex items-center justify-between p-4 border rounded-lg bg-card hover:border-primary/50 transition-colors cursor-pointer group">
      <div>
        <h4 className="font-bold text-lg group-hover:text-primary transition-colors">{title}</h4>
        <p className="text-muted-foreground text-sm">{desc}</p>
      </div>
      <div className="text-right">
        <span className="block font-bold text-xl">${price}</span>
        <Button size="sm" variant="ghost" className="h-8">{btnText || "Select"}</Button>
      </div>
    </div>
  );
}
