import { Navbar } from "@/components/layout/Navbar";
import { Footer } from "@/components/layout/Footer";
import { Button } from "@/components/ui/button";
import { heroBg, influencers, stats } from "@/lib/mock-data";
import { InfluencerCard } from "@/components/ui/influencer-card";
import { Link } from "wouter";
import { motion } from "framer-motion";
import { ArrowRight, Zap, Shield, TrendingUp, Users } from "lucide-react";
import { useLanguage } from "@/context/LanguageContext";

export default function LandingPage() {
  const { t } = useLanguage();

  return (
    <div className="min-h-screen flex flex-col bg-background">
      <Navbar />
      
      {/* Hero Section */}
      <section className="relative pt-20 pb-32 overflow-hidden">
        <div className="absolute inset-0 z-0 opacity-10 dark:opacity-20 pointer-events-none">
          <img src={heroBg} alt="Background" className="w-full h-full object-cover" />
          <div className="absolute inset-0 bg-gradient-to-b from-background/0 via-background/50 to-background" />
        </div>

        <div className="container mx-auto px-4 relative z-10">
          <div className="flex flex-col lg:flex-row items-center gap-12">
            <div className="flex-1 space-y-8">
              <motion.div 
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.5 }}
              >
                <div className="inline-flex items-center gap-2 px-3 py-1 rounded-full bg-primary/10 text-primary text-sm font-medium mb-6 border border-primary/20">
                  <Zap className="size-3 fill-primary" />
                  <span>{t('hero.badge')}</span>
                </div>
                <h1 className="text-5xl lg:text-7xl font-display font-extrabold leading-tight tracking-tight text-foreground mb-6">
                  {t('hero.title')} <br/>
                  <span className="gradient-text">{t('hero.titleHighlight')}</span>
                </h1>
                <p className="text-xl text-muted-foreground max-w-xl leading-relaxed">
                  {t('hero.subtitle')}
                </p>
              </motion.div>

              <motion.div 
                initial={{ opacity: 0, y: 20 }}
                animate={{ opacity: 1, y: 0 }}
                transition={{ duration: 0.5, delay: 0.2 }}
                className="flex flex-wrap gap-4"
              >
                <Link href="/marketplace">
                  <Button size="lg" className="h-14 px-8 text-lg shadow-xl shadow-primary/25 rounded-full">
                    {t('hero.ctaMarketplace')} <ArrowRight className="ml-2 size-5" />
                  </Button>
                </Link>
                <Link href="/auth?tab=register">
                  <Button size="lg" variant="outline" className="h-14 px-8 text-lg bg-white/50 backdrop-blur-sm rounded-full border-primary/20 hover:bg-white/80">
                    {t('hero.ctaInfluencer')}
                  </Button>
                </Link>
              </motion.div>

              <motion.div 
                initial={{ opacity: 0 }}
                animate={{ opacity: 1 }}
                transition={{ duration: 0.5, delay: 0.4 }}
                className="flex items-center gap-8 pt-4"
              >
                <div>
                  <p className="font-bold text-2xl text-foreground">{stats.totalInfluencers}</p>
                  <p className="text-sm text-muted-foreground">{t('hero.statsInfluencers')}</p>
                </div>
                <div className="w-px h-10 bg-border" />
                <div>
                  <p className="font-bold text-2xl text-foreground">{stats.campaignsCompleted}</p>
                  <p className="text-sm text-muted-foreground">{t('hero.statsCampaigns')}</p>
                </div>
                <div className="w-px h-10 bg-border" />
                <div>
                  <p className="font-bold text-2xl text-foreground">{stats.payoutsProcessed}</p>
                  <p className="text-sm text-muted-foreground">{t('hero.statsPayouts')}</p>
                </div>
              </motion.div>
            </div>

            <div className="flex-1 relative w-full max-w-xl lg:max-w-none">
              <motion.div 
                 initial={{ opacity: 0, scale: 0.9 }}
                 animate={{ opacity: 1, scale: 1 }}
                 transition={{ duration: 0.7 }}
                 className="relative z-10"
              >
                 <div className="grid grid-cols-2 gap-4">
                    <div className="space-y-4 mt-12">
                       <InfluencerCard influencer={influencers[0]} />
                       <InfluencerCard influencer={influencers[2]} />
                    </div>
                    <div className="space-y-4">
                       <InfluencerCard influencer={influencers[1]} />
                       <InfluencerCard influencer={influencers[3]} />
                    </div>
                 </div>

                 {/* Floating Element */}
                 <motion.div 
                    animate={{ y: [0, -10, 0] }}
                    transition={{ duration: 4, repeat: Infinity, ease: "easeInOut" }}
                    className="absolute top-1/2 left-1/2 -translate-x-1/2 -translate-y-1/2 bg-white dark:bg-slate-900 p-4 rounded-2xl shadow-2xl border border-border/50 backdrop-blur-xl z-30"
                 >
                    <div className="flex items-center gap-3">
                      <div className="size-10 rounded-full bg-green-100 flex items-center justify-center">
                        <TrendingUp className="size-6 text-green-600" />
                      </div>
                      <div>
                        <p className="text-xs text-muted-foreground font-medium">AI Match Score</p>
                        <p className="text-xl font-bold text-foreground">98.5%</p>
                      </div>
                    </div>
                 </motion.div>
              </motion.div>
            </div>
          </div>
        </div>
      </section>

      {/* Features Section */}
      <section className="py-24 bg-secondary/30 border-y border-border/50">
        <div className="container mx-auto px-4">
          <div className="text-center max-w-2xl mx-auto mb-16">
            <h2 className="text-3xl font-display font-bold mb-4">Why InfluencerHub?</h2>
            <p className="text-muted-foreground text-lg">We've reimagined influencer marketing with technology at the core. No more spreadsheets, no more uncertainty.</p>
          </div>

          <div className="grid md:grid-cols-3 gap-8">
            <FeatureCard 
              icon={<Zap className="size-6 text-yellow-500" />}
              title="AI Recommendation Engine"
              description="Our Python-based algorithm analyzes millions of data points to predict which influencers will actually drive sales for your specific niche."
            />
            <FeatureCard 
              icon={<Shield className="size-6 text-blue-500" />}
              title="Secure Escrow Payments"
              description="Funds are held safely in escrow until the work is delivered and approved. We support USD, Credit Cards, and USDT."
            />
            <FeatureCard 
              icon={<Users className="size-6 text-purple-500" />}
              title="Verified Profiles Only"
              description="Every influencer passes a strict KYC process and audience quality audit. Say goodbye to fake followers and bots."
            />
          </div>
        </div>
      </section>

      {/* CTA Section */}
      <section className="py-24 relative overflow-hidden">
         <div className="absolute inset-0 bg-primary z-0">
           <div className="absolute inset-0 bg-[url('https://www.transparenttextures.com/patterns/cubes.png')] opacity-10 mix-blend-overlay" />
           <div className="absolute -top-24 -right-24 size-96 bg-white/10 rounded-full blur-3xl" />
           <div className="absolute -bottom-24 -left-24 size-96 bg-black/10 rounded-full blur-3xl" />
         </div>
         
         <div className="container mx-auto px-4 relative z-10 text-center text-white">
           <h2 className="text-4xl md:text-5xl font-display font-bold mb-6">Ready to launch your campaign?</h2>
           <p className="text-xl text-white/80 max-w-2xl mx-auto mb-8">Join thousands of brands and creators who are already scaling with InfluencerHub AI.</p>
           <div className="flex justify-center gap-4">
             <Link href="/auth?tab=register">
               <Button size="lg" variant="secondary" className="h-14 px-8 text-lg rounded-full shadow-xl">
                 Get Started for Free
               </Button>
             </Link>
             <Link href="/marketplace">
               <Button size="lg" variant="outline" className="h-14 px-8 text-lg rounded-full bg-transparent text-white border-white/30 hover:bg-white/10 hover:text-white">
                 Browse Influencers
               </Button>
             </Link>
           </div>
         </div>
      </section>

      <Footer />
    </div>
  );
}

function FeatureCard({ icon, title, description }: { icon: any, title: string, description: string }) {
  return (
    <div className="bg-card p-8 rounded-2xl border border-border/50 shadow-sm hover:shadow-md transition-all">
      <div className="size-12 rounded-xl bg-secondary flex items-center justify-center mb-6">
        {icon}
      </div>
      <h3 className="text-xl font-bold font-display mb-3">{title}</h3>
      <p className="text-muted-foreground leading-relaxed">
        {description}
      </p>
    </div>
  );
}
