import { useState } from "react";
import { Navbar } from "@/components/layout/Navbar";
import { influencers } from "@/lib/mock-data";
import { InfluencerCard } from "@/components/ui/influencer-card";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Slider } from "@/components/ui/slider";
import { Checkbox } from "@/components/ui/checkbox";
import { 
  Search, 
  Filter, 
  SlidersHorizontal,
  ArrowUpDown,
  ShieldCheck
} from "lucide-react";
import {
  Select,
  SelectContent,
  SelectItem,
  SelectTrigger,
  SelectValue,
} from "@/components/ui/select";
import { Button } from "@/components/ui/button";
import { Separator } from "@/components/ui/separator";
import { Sheet, SheetContent, SheetTrigger } from "@/components/ui/sheet";
import { useLanguage } from "@/context/LanguageContext";

export default function MarketplacePage() {
  const { t } = useLanguage();
  const [priceRange, setPriceRange] = useState([0, 5000]);
  const [searchQuery, setSearchQuery] = useState("");

  const filteredInfluencers = influencers.filter(inf => {
    const matchesSearch = inf.name.toLowerCase().includes(searchQuery.toLowerCase()) || 
                          inf.niche.toLowerCase().includes(searchQuery.toLowerCase());
    const matchesPrice = inf.price >= priceRange[0] && inf.price <= priceRange[1];
    return matchesSearch && matchesPrice;
  });

  return (
    <div className="min-h-screen bg-background flex flex-col">
      <Navbar />
      
      <main className="flex-1 container mx-auto px-4 py-8">
        <div className="flex flex-col md:flex-row gap-8">
          
          {/* Filters Sidebar (Desktop) */}
          <aside className="hidden lg:block w-64 shrink-0 space-y-8 sticky top-24 h-fit">
            <div>
              <h3 className="font-display font-bold text-lg mb-4 flex items-center gap-2">
                <Filter className="size-4" /> {t('marketplace.filters')}
              </h3>
              
              <div className="space-y-6">
                <div className="space-y-4">
                  <Label>{t('marketplace.niche')}</Label>
                  <div className="space-y-2">
                    {["Fashion", "Tech", "Gaming", "Travel", "Fitness", "Crypto"].map((niche) => (
                      <div key={niche} className="flex items-center space-x-2">
                        <Checkbox id={niche} />
                        <label
                          htmlFor={niche}
                          className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                        >
                          {niche}
                        </label>
                      </div>
                    ))}
                  </div>
                </div>

                <Separator />

                <div className="space-y-4">
                  <Label>{t('marketplace.platform')}</Label>
                  <div className="space-y-2">
                    {["Instagram", "TikTok", "YouTube", "Twitch", "Twitter"].map((platform) => (
                      <div key={platform} className="flex items-center space-x-2">
                        <Checkbox id={platform} />
                        <label
                          htmlFor={platform}
                          className="text-sm font-medium leading-none peer-disabled:cursor-not-allowed peer-disabled:opacity-70"
                        >
                          {platform}
                        </label>
                      </div>
                    ))}
                  </div>
                </div>

                <Separator />

                <div className="space-y-4">
                  <div className="flex justify-between">
                    <Label>{t('marketplace.priceRange')}</Label>
                    <span className="text-xs text-muted-foreground">${priceRange[0]} - ${priceRange[1]}+</span>
                  </div>
                  <Slider
                    defaultValue={[0, 5000]}
                    max={10000}
                    step={100}
                    value={priceRange}
                    onValueChange={setPriceRange}
                    className="mt-2"
                  />
                </div>
              </div>
            </div>
          </aside>

          {/* Main Content */}
          <div className="flex-1">
            {/* Header / Controls */}
            <div className="flex flex-col sm:flex-row gap-4 mb-8 justify-between items-start sm:items-center">
              <div>
                <h1 className="text-3xl font-display font-bold text-foreground flex items-center gap-2">
                   {t('marketplace.title')}
                   <ShieldCheck className="size-6 text-primary" />
                </h1>
                <p className="text-muted-foreground">{t('marketplace.subtitle')}</p>
              </div>

              <div className="flex flex-wrap gap-2 w-full sm:w-auto">
                <div className="relative w-full sm:w-64">
                  <Search className="absolute left-2.5 top-2.5 h-4 w-4 text-muted-foreground" />
                  <Input
                    type="search"
                    placeholder={t('marketplace.searchPlaceholder')}
                    className="pl-8"
                    value={searchQuery}
                    onChange={(e) => setSearchQuery(e.target.value)}
                  />
                </div>
                
                <Sheet>
                  <SheetTrigger asChild>
                    <Button variant="outline" size="icon" className="lg:hidden">
                      <SlidersHorizontal className="size-4" />
                    </Button>
                  </SheetTrigger>
                  <SheetContent side="left">
                    <div className="py-4">
                      <h3 className="font-bold mb-4">{t('marketplace.filters')}</h3>
                      <p className="text-muted-foreground">Mobile filters...</p>
                    </div>
                  </SheetContent>
                </Sheet>

                <Select defaultValue="relevance">
                  <SelectTrigger className="w-[160px]">
                    <div className="flex items-center gap-2">
                      <ArrowUpDown className="size-3.5" />
                      <SelectValue placeholder={t('marketplace.sort')} />
                    </div>
                  </SelectTrigger>
                  <SelectContent>
                    <SelectItem value="relevance">{t('marketplace.relevance')}</SelectItem>
                    <SelectItem value="price_asc">{t('marketplace.priceLowHigh')}</SelectItem>
                    <SelectItem value="price_desc">{t('marketplace.priceHighLow')}</SelectItem>
                    <SelectItem value="followers">{t('marketplace.followers')}</SelectItem>
                  </SelectContent>
                </Select>
              </div>
            </div>

            {/* Grid */}
            {filteredInfluencers.length > 0 ? (
              <div className="grid grid-cols-1 md:grid-cols-2 xl:grid-cols-3 gap-6">
                {filteredInfluencers.map((inf) => (
                  <InfluencerCard key={inf.id} influencer={inf} />
                ))}
              </div>
            ) : (
              <div className="text-center py-20 border rounded-xl border-dashed">
                <p className="text-lg text-muted-foreground">{t('marketplace.noResults')}</p>
                <Button variant="link" onClick={() => {setSearchQuery(""); setPriceRange([0, 5000])}}>{t('marketplace.clearFilters')}</Button>
              </div>
            )}
          </div>
        </div>
      </main>
    </div>
  );
}
