import { DashboardLayout } from "@/components/layout/DashboardLayout";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Input } from "@/components/ui/input";
import { Label } from "@/components/ui/label";
import { Switch } from "@/components/ui/switch";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Avatar, AvatarFallback, AvatarImage } from "@/components/ui/avatar";
import { useAuth } from "@/context/AuthContext";
import { useLanguage } from "@/context/LanguageContext";
import { Bell, Lock, Shield, User, Smartphone, Mail, Upload, CheckCircle } from "lucide-react";

export default function SettingsPage() {
  const { user } = useAuth();
  const { t } = useLanguage();

  if (!user) return null;

  return (
    <DashboardLayout>
      <div className="mb-8">
        <h1 className="text-3xl font-display font-bold">{t('settings.title')}</h1>
        <p className="text-muted-foreground">{t('settings.subtitle')}</p>
      </div>

      <Tabs defaultValue="profile" className="space-y-6">
        <TabsList className="w-full md:w-auto">
          <TabsTrigger value="profile">{t('settings.profile')}</TabsTrigger>
          <TabsTrigger value="verification">{t('settings.verification')}</TabsTrigger>
          <TabsTrigger value="notifications">{t('settings.notifications')}</TabsTrigger>
          <TabsTrigger value="security">{t('settings.security')}</TabsTrigger>
        </TabsList>

        <TabsContent value="profile">
          <Card>
            <CardHeader>
              <CardTitle>{t('settings.publicProfile')}</CardTitle>
              <CardDescription>{t('settings.publicProfileDesc')}</CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-center gap-6">
                <Avatar className="size-24">
                  <AvatarImage src={user.avatar} />
                  <AvatarFallback className="text-2xl">{user.name.charAt(0)}</AvatarFallback>
                </Avatar>
                <Button variant="outline">{t('settings.changeAvatar')}</Button>
              </div>
              
              <div className="grid md:grid-cols-2 gap-4">
                <div className="space-y-2">
                  <Label>{t('settings.displayName')}</Label>
                  <Input defaultValue={user.name} />
                </div>
                <div className="space-y-2">
                  <Label>{t('settings.email')}</Label>
                  <Input defaultValue={user.email} disabled />
                </div>
                <div className="space-y-2 md:col-span-2">
                   <Label>{t('settings.bio')}</Label>
                   <Input defaultValue={user.role === 'influencer' ? "Professional content creator..." : "Marketing Director at TechNova"} />
                </div>
              </div>
              
              <div className="flex justify-end">
                <Button>{t('settings.save')}</Button>
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="verification">
          <Card>
            <CardHeader>
              <CardTitle className="flex items-center gap-2">
                {t('settings.kyc')}
                <span className="text-xs bg-green-100 text-green-700 px-2 py-1 rounded-full border border-green-200 font-medium">
                  {t('settings.verifiedStatus')}
                </span>
              </CardTitle>
              <CardDescription>
                To ensure trust on InfluencerHub, all users must complete identity verification.
              </CardDescription>
            </CardHeader>
            <CardContent className="space-y-8">
               <div className="border rounded-xl p-6 bg-secondary/10">
                 <div className="flex items-start gap-4">
                   <div className="bg-green-100 p-3 rounded-full">
                     <CheckCircle className="size-6 text-green-600" />
                   </div>
                   <div>
                     <h3 className="font-bold text-lg">Identity Verified</h3>
                     <p className="text-muted-foreground text-sm mt-1">
                       Your government ID has been verified. You now have the "Verified" badge on your profile.
                     </p>
                     <p className="text-xs text-muted-foreground mt-2">Verified on: 12 Jan 2024</p>
                   </div>
                 </div>
               </div>

               <div className="space-y-4 opacity-50 pointer-events-none">
                 <Label>{t('settings.uploadId')}</Label>
                 <div className="border-2 border-dashed rounded-xl p-12 text-center">
                   <Upload className="size-8 text-muted-foreground mx-auto mb-4" />
                   <p className="text-sm font-medium">{t('settings.uploadDesc')}</p>
                   <Button variant="secondary" className="mt-4">Select File</Button>
                 </div>
               </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="notifications">
          <Card>
            <CardHeader>
              <CardTitle>{t('settings.notifications')}</CardTitle>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label className="text-base">{t('settings.emailNotif')}</Label>
                  <p className="text-sm text-muted-foreground">{t('settings.emailNotifDesc')}</p>
                </div>
                <Switch defaultChecked />
              </div>
              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label className="text-base">{t('settings.marketing')}</Label>
                  <p className="text-sm text-muted-foreground">{t('settings.marketingDesc')}</p>
                </div>
                <Switch />
              </div>
              <div className="flex items-center justify-between">
                <div className="space-y-0.5">
                  <Label className="text-base">{t('settings.sms')}</Label>
                  <p className="text-sm text-muted-foreground">{t('settings.smsDesc')}</p>
                </div>
                <Switch defaultChecked />
              </div>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="security">
          <Card>
            <CardHeader>
              <CardTitle>{t('settings.security')}</CardTitle>
              <CardDescription>Protect your account with additional security layers.</CardDescription>
            </CardHeader>
            <CardContent className="space-y-6">
              <div className="space-y-4">
                <div className="flex items-center justify-between p-4 border rounded-lg bg-secondary/10">
                  <div className="flex items-center gap-4">
                    <div className="bg-primary/10 p-2 rounded-full">
                      <Lock className="size-5 text-primary" />
                    </div>
                    <div>
                      <p className="font-bold">{t('settings.twoFactor')}</p>
                      <p className="text-sm text-muted-foreground">{t('settings.twoFactorDesc')}</p>
                    </div>
                  </div>
                  <Button variant="outline">{t('settings.enable')}</Button>
                </div>
              </div>
              
              <div className="pt-4 border-t">
                <h4 className="text-sm font-bold mb-4">{t('settings.activeSessions')}</h4>
                <div className="flex items-center justify-between text-sm">
                   <div className="flex items-center gap-2">
                     <Smartphone className="size-4 text-muted-foreground" />
                     <span>iPhone 14 Pro - San Francisco, US</span>
                   </div>
                   <span className="text-green-500 text-xs">{t('settings.activeNow')}</span>
                </div>
              </div>
            </CardContent>
          </Card>
        </TabsContent>
      </Tabs>
    </DashboardLayout>
  );
}
