import { DashboardLayout } from "@/components/layout/DashboardLayout";
import { useLanguage } from "@/context/LanguageContext";
import { Card, CardContent, CardHeader, CardTitle, CardDescription } from "@/components/ui/card";
import { Button } from "@/components/ui/button";
import { Table, TableBody, TableCell, TableHead, TableHeader, TableRow } from "@/components/ui/table";
import { Badge } from "@/components/ui/badge";
import { Tabs, TabsContent, TabsList, TabsTrigger } from "@/components/ui/tabs";
import { Wallet, ArrowUpRight, ArrowDownLeft, CreditCard, Bitcoin, Building } from "lucide-react";

export default function WalletPage() {
  const { t } = useLanguage();

  const transactions = [
    { id: "TX-9823", type: "deposit", method: "USDT", amount: 5000, status: "completed", date: "2024-05-20" },
    { id: "TX-9822", type: "payment", method: "Escrow - Campaña #442", amount: -1200, status: "held", date: "2024-05-19" },
    { id: "TX-9821", type: "deposit", method: "Stripe", amount: 2000, status: "completed", date: "2024-05-15" },
    { id: "TX-9810", type: "withdrawal", method: "PayPal", amount: -500, status: "completed", date: "2024-05-10" },
  ];

  return (
    <DashboardLayout>
      <div className="mb-8">
        <h1 className="text-3xl font-display font-bold">{t('wallet.title')}</h1>
        <p className="text-muted-foreground">Manage your funds, deposits, and withdrawals securely.</p>
      </div>

      <div className="grid md:grid-cols-3 gap-8 mb-8">
        {/* Main Balance Card */}
        <Card className="md:col-span-2 bg-gradient-to-br from-primary/90 to-primary text-primary-foreground border-none shadow-xl">
          <CardContent className="p-8">
             <div className="flex justify-between items-start mb-8">
               <div>
                 <p className="text-primary-foreground/80 font-medium mb-1">{t('wallet.balance')}</p>
                 <h2 className="text-5xl font-bold tracking-tight">$5,300.00</h2>
               </div>
               <div className="bg-white/20 p-3 rounded-full">
                 <Wallet className="size-8 text-white" />
               </div>
             </div>
             
             <div className="grid grid-cols-2 gap-8">
               <div>
                 <p className="text-sm text-primary-foreground/70 mb-1">{t('wallet.available')}</p>
                 <p className="text-xl font-bold">$4,100.00</p>
               </div>
               <div>
                 <p className="text-sm text-primary-foreground/70 mb-1">{t('wallet.pending')}</p>
                 <p className="text-xl font-bold">$1,200.00</p>
               </div>
             </div>
          </CardContent>
        </Card>

        {/* Actions Card */}
        <Card className="flex flex-col justify-center space-y-4 p-6">
          <Button size="lg" className="w-full h-14 text-lg flex items-center gap-2 shadow-lg">
             <ArrowDownLeft className="size-5" /> {t('wallet.deposit')}
          </Button>
          <Button size="lg" variant="outline" className="w-full h-14 text-lg flex items-center gap-2">
             <ArrowUpRight className="size-5" /> {t('wallet.withdraw')}
          </Button>
        </Card>
      </div>

      {/* Payment Methods & History */}
      <Tabs defaultValue="history" className="space-y-6">
        <TabsList>
          <TabsTrigger value="history">{t('wallet.history')}</TabsTrigger>
          <TabsTrigger value="methods">Payment Methods</TabsTrigger>
        </TabsList>

        <TabsContent value="history">
          <Card>
            <CardHeader>
              <CardTitle>Recent Transactions</CardTitle>
              <CardDescription>All your financial activity in one place.</CardDescription>
            </CardHeader>
            <CardContent>
              <Table>
                <TableHeader>
                  <TableRow>
                    <TableHead>ID</TableHead>
                    <TableHead>Type</TableHead>
                    <TableHead>Method</TableHead>
                    <TableHead>Date</TableHead>
                    <TableHead>Status</TableHead>
                    <TableHead className="text-right">Amount</TableHead>
                  </TableRow>
                </TableHeader>
                <TableBody>
                  {transactions.map((tx) => (
                    <TableRow key={tx.id}>
                      <TableCell className="font-mono text-xs">{tx.id}</TableCell>
                      <TableCell className="capitalize">{tx.type}</TableCell>
                      <TableCell>{tx.method}</TableCell>
                      <TableCell>{tx.date}</TableCell>
                      <TableCell>
                        <StatusBadge status={tx.status} />
                      </TableCell>
                      <TableCell className={`text-right font-bold ${tx.amount > 0 ? 'text-green-600' : 'text-foreground'}`}>
                        {tx.amount > 0 ? '+' : ''}{tx.amount.toLocaleString()} USD
                      </TableCell>
                    </TableRow>
                  ))}
                </TableBody>
              </Table>
            </CardContent>
          </Card>
        </TabsContent>

        <TabsContent value="methods">
           <div className="grid md:grid-cols-3 gap-4">
             <MethodCard icon={<CreditCard className="size-6" />} title="Stripe / Cards" desc="Instant deposits, 2.9% fee" />
             <MethodCard icon={<Bitcoin className="size-6 text-green-500" />} title="USDT (TRC20)" desc="Low fees, crypto friendly" />
             <MethodCard icon={<Building className="size-6 text-blue-500" />} title="Bank Transfer" desc="For amounts > $10k" />
           </div>
        </TabsContent>
      </Tabs>
    </DashboardLayout>
  );
}

function StatusBadge({ status }: { status: string }) {
  const styles = {
    completed: "bg-green-100 text-green-700 hover:bg-green-100/80",
    held: "bg-yellow-100 text-yellow-700 hover:bg-yellow-100/80",
    pending: "bg-gray-100 text-gray-700 hover:bg-gray-100/80",
  };
  return (
    <Badge variant="secondary" className={styles[status as keyof typeof styles] || styles.pending}>
      {status}
    </Badge>
  );
}

function MethodCard({ icon, title, desc }: any) {
  return (
    <Card className="cursor-pointer hover:border-primary transition-all">
      <CardContent className="p-6 flex items-center gap-4">
        <div className="size-12 rounded-full bg-secondary flex items-center justify-center">
          {icon}
        </div>
        <div>
          <h4 className="font-bold">{title}</h4>
          <p className="text-sm text-muted-foreground">{desc}</p>
        </div>
      </CardContent>
    </Card>
  );
}
